function Diffraction(varargin)

% Cette fonction permet de simuler la reconstruction, neutron par neutron, 
%   de la figure de diffraction donne par une fente. 
%   Concrtement, on considre l'exprience de Zeilinger (Rev. Mod. Phys,
%   60, 1988: single- and double-slit diffraction of neutrons)
%   >>nobjetmax = Nombre d'objets physique (ici des neutrons)  envoyer successivement
%
% Olivier Pujol (janvier 2013)

%% Vrifications diverses

if (length(varargin)>1), error('Mauvais nombre d''arguments d''entre');end
if (isempty(varargin)),nobjetmax=1000;end
if (length(varargin)==1),nobjetmax=varargin{1};end

%% Constantes

epsilon=90E-6;      % Largeur d'une fente (m)
lambdaDB=1.926E-9;  % Longueur d'onde du rayonnement (m)
d=5;                % Distance fente-cran (m)

%% Domaine spatial sur l'cran

x=linspace(-2E-4,2E-4,2500);        % Perpendiculaire  la fente
y=linspace(-0.1,0.1,2);             % Parallle  la fente

%% Intensit normalise ou probabilit d'impact

% Amplitude normalise
amplitude=(sin(pi*epsilon.*x/(lambdaDB*d))./(pi*epsilon.*x/(lambdaDB*d))); 
        
% Probabilit    
probabilite=amplitude.^2;

%% Partie graphique 

figure(1)

% Enlever les deux commentaires (%) pour faire un film
%film(1:nneutronmax) = struct('cdata', [],'colormap', []);
%set(gca,'nextplot','replacechildren');

% Reprsentation de la probabilit linique normalise
subplot(3,1,1)
plot(x*1E6,probabilite,'b'); % Axe des x en m (commodit)
axis([min(x*1E6) max(x*1E6) 0 1]);
set(gca,'XTick',-200:100:200,'XTickLabel',{'-200','','0','','200'},...
    'YTick',[0 0.5 1],'YTickLabel',{'0','','1'},'Fontsize',16);
xlabel('{\it x} (m)','Fontsize',16);
ylabel('{\it{\rho}_p / \rho_{p,max}}','Fontsize',16);

% Graphe  deux dimensions des interfrences sur l'cran
[xx,yy]=meshgrid(x,y);  
probabilite2D=repmat(probabilite,length(y),1);
subplot(3,1,2)
pcolor(xx*1E6,yy,probabilite2D) % Axe des x en m (commodit)
axis([min(min(xx*1E6)) max(max(xx*1E6)) min(min(yy)) max(max(yy))]);
colormap(gray);shading interp
set(gca,'XTick',-200:100:200,'XTickLabel',{'-200','','0','','200'},...
    'YTick',[],'Fontsize',16);
xlabel('{\it x} (m)','Fontsize',16);

% Reconstruction de la figure prcdente lectron par lectron
subplot(3,1,3)
set(gca,'Color','k');
set(gca,'XTick',-200:100:200,'XTickLabel',{'-200','','0','','200'},...
    'YTick',[],'Fontsize',16);
axis([min(x*1E6) max(x*1E6) min(y) max(y)]);
xlabel('{\it x} (m)','Fontsize',16);
set(gcf, 'InvertHardCopy', 'off');
hold on

nobjet=1;          % Initialisation du nombre de lectrons

% Initialisation des vecteurs pour un histogramme du nombre d'impact 
nintervalle=100;   
xhist=linspace(min(x),max(x),nintervalle);
nhist=zeros(length(xhist),1);

while nobjet<=nobjetmax
    xal= min(x) + (max(x)-min(x)).*rand(1,1);   % Coordonne x alatoire
    yal= min(y) + (max(y)-min(y)).*rand(1,1);   % Coordonne y alatoire
    
    ampimp=sin(pi*epsilon*xal/(lambdaDB*d))/(pi*epsilon*xal/(lambdaDB*d));
    probaimp=ampimp^2;
    
    nbal=rand(1,1);     % Nombre alatoire compris entre 0 et 1   
    if probaimp > nbal
        plot(xal*1E6,yal,'wo','markerfacecolor','w','markersize',3);    % Axe des x en m (commodit)
        hold on
        % On repre l'indice de xal dans le vecteur xhist puis on incrmente
        % le compteur nhist correspondant
        ixal=round(1+((xal-min(xhist))/(max(xhist)-min(xhist)))*(100-1));
        nhist(ixal)=nhist(ixal)+1;
        % On incrmente le numro de l'objet physique envoy
        nobjet=nobjet+1;    % Mettre un commentaire (%) pour faire un film
    end
    pause(0.01)

    % Enlever les deux commentaires (%) ci-dessous pour faire un film
    %nobjet=nobjet+1;     
    %film(nneutron)=getframe(gcf);  % Enlever le commentaire (%) pour faire un film

end

% Enlever le commentaire (%) pour faire un film
%movie2avi(film, 'Diffraction.avi','fps',15,'compression','Cinepak');

figure(2)

% Reprsentation de l'intensit normalise
subplot(2,1,1)
plot(x*1E6,probabilite,'b'); % Axe des x en m (commodit)
axis([min(x*1E6) max(x*1E6) 0 1]);
set(gca,'XTick',-200:100:200,'XTickLabel',{'-200','','0','','200'},...
    'YTick',[0 0.5 1],'YTickLabel',{'0','','1'},'Fontsize',16);
xlabel('{\it x} (m)','Fontsize',16);
ylabel('{\it{\rho}_p / \rho_{p,max}}','Fontsize',16);

% Histogramme du nombre d'impact
subplot(2,1,2)
bar(xhist*1E6,nhist/max(nhist),'k'); % Axe des x en m (commodit)
axis([min(xhist*1E6) max(xhist*1E6) 0 1]);
set(gca,'XTick',-200:100:200,'XTickLabel',{'-200','','0','','200'},...
    'YTick',[0 0.5 1],'YTickLabel',{'0','','1'},'Fontsize',16);
xlabel('{\it x} (m)','Fontsize',16);
ylabel('{\it N / N_{max}}','Fontsize',16);

somme=sum(nhist);
display(['Nombre de neutrons envoys : ',num2str(somme)]);

return